import nodemailer from 'nodemailer';
import FileStorage from '../utils/fileStorage.js';

/**
 * Email service for sending newsletters and verification emails
 */
class EmailService {
  static transporter = null;
  static isConnected = false;

  /**
   * Initialize the email transporter
   */
  static async initialize() {
    try {
      if (!process.env.EMAIL_USER || !process.env.EMAIL_PASSWORD) {
        console.error('❌ Missing EMAIL_USER or EMAIL_PASSWORD in .env');
        this.isConnected = false;
        return false;
      }

      this.transporter = nodemailer.createTransport({
        host: process.env.EMAIL_HOST || 'smtp.gmail.com',
        port: parseInt(process.env.EMAIL_PORT) || 465,
        secure: true,
        auth: {
          user: process.env.EMAIL_USER,
          pass: process.env.EMAIL_PASSWORD,
        },
      });

      await this.transporter.verify();
      this.isConnected = true;
      console.log('✅ Email server connected successfully');
      return true;
    } catch (error) {
      this.isConnected = false;
      console.error('❌ Email connection failed:', error.message);
      return false;
    }
  }

  /**
   * Private method to send an email
   */
  static async #sendEmail(to, subject, html) {
    if (!this.isConnected || !this.transporter) {
      console.warn(`⚠️ Cannot send email - service not connected`);
      throw new Error('Email service not available');
    }

    try {
      const info = await this.transporter.sendMail({
        from: `"${process.env.COMPANY_NAME || 'TLC Homes'}" <${process.env.EMAIL_USER}>`,
        to,
        subject,
        html,
      });
      console.log(`✅ Email sent to ${to}`);
      return info;
    } catch (error) {
      console.error(`❌ Failed to send email to ${to}:`, error.message);
      throw error;
    }
  }

  /**
 * Send verification email to new subscriber
 */
  static async sendVerificationEmail(email, id, firstName) {
    const verificationUrl = `${process.env.URL}/api/subscriber/verify?id=${id}&email=${encodeURIComponent(email)}`;

    const html = `
    <!DOCTYPE html>
    <html lang="en">
      <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <meta http-equiv="X-UA-Compatible" content="IE=edge">
        <title>Verify Your Email - TLC Homes</title>
        <style>
          body { 
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif; 
            line-height: 1.6; 
            color: #333; 
            margin: 0;
            padding: 0;
            background-color: #f5f5f5;
          }
          .wrapper {
            width: 100%;
            background-color: #f5f5f5;
            padding: 20px 0;
          }
          .container { 
            max-width: 600px; 
            margin: 0 auto; 
            background-color: #ffffff;
            overflow: hidden;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.08);
          }
          .header { 
            background-color: #000; 
            color: #fff; 
            padding: 40px 20px; 
            text-align: center; 
            border-bottom: 3px solid #d0c82a;
          }
          .header h1 {
            margin: 0;
            font-size: 24px;
            font-weight: 700;
            letter-spacing: -0.5px;
          }
          .content { 
            padding: 50px 40px; 
            background-color: #ffffff; 
          }
          .content h2 {
            margin: 0 0 20px 0;
            font-size: 22px;
            color: #000000;
            font-weight: 600;
          }
          .content p {
            margin: 0 0 18px 0;
            font-size: 16px;
            color: #333333;
            line-height: 1.7;
          }
          .button-container {
            text-align: center;
            margin: 40px 0;
          }
          .link-fallback {
            margin-top: 30px;
            padding: 20px;
            background-color: #fafafa;
            border-left: 3px solid #000;
          }
          .link-fallback p {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #666666;
          }
          .footer { 
            padding: 30px 20px; 
            text-align: center; 
            font-size: 13px; 
            color: #999999;
            background-color: #fafafa;
            border-top: 1px solid #e5e5e5;
          }
          .footer p {
            margin: 8px 0;
          }
          @media only screen and (max-width: 600px) {
            .content {
              padding: 40px 24px;
            }
            .header {
              padding: 32px 20px;
            }
          }
        </style>
      </head>
      <body>
        <div class="wrapper">
          <div class="container">
            <div class="header">
              <h1>Welcome to TLC Homes Newsletter</h1>
            </div>
            <div class="content">
              <h2>Hi ${firstName},</h2>
              <p>Thank you for subscribing to TLC Homes Regional Insights!</p>
              <p>Please verify your email address to start receiving our latest updates on real estate trends, market insights, and property news.</p>
              
              <table width="100%" cellpadding="0" cellspacing="0" border="0" style="margin: 40px 0;">
                <tr>
                  <td align="center">
                    <table cellpadding="0" cellspacing="0" border="0">
                      <tr>
                        <td align="center" style="background-color: #000000; padding: 18px 48px;">
                          <a href="${verificationUrl}" target="_blank" style="color: #ffffff; text-decoration: none; font-size: 16px; font-weight: 600; letter-spacing: 0.3px; display: inline-block;">
                            Verify Your Email
                          </a>
                        </td>
                      </tr>
                    </table>
                  </td>
                </tr>
              </table>

              <p>If the button doesn't work, copy and paste this link into your browser:</p>
              <p style="word-break: break-all; color: #d0c82a; font-family: monospace; font-size: 14px;">${verificationUrl}</p>
              
              <p style="margin-top: 30px; font-size: 14px; color: #666; border-top: 1px solid #e5e5e5; padding-top: 20px;">
                <em>This verification link will expire in 24 hours.</em>
              </p>
            </div>
            <div class="footer">
              <p>&copy; ${new Date().getFullYear()} TLC Homes Services. All rights reserved.</p>
              <p>You received this email because you signed up for our newsletter.</p>
            </div>
          </div>
        </div>
      </body>
    </html>
  `;

    await this.#sendEmail(email, 'Verify Your Email - TLC Homes Newsletter', html);
  }

  /**
   * Send blog notification email with enhanced template
   */
  static async sendBlogNotification(blogData) {
    if (!this.isConnected) {
      console.warn('⚠️ Cannot send blog notifications - email service not connected');
      return {
        success: false,
        sent: 0,
        failed: 0,
        total: 0,
        message: 'Email service not connected'
      };
    }

    try {
      const subscribers = await FileStorage.getSubscribers();

      if (subscribers.length === 0) {
        console.log('🔭 No verified subscribers to notify');
        return { success: true, sent: 0, failed: 0, total: 0, message: 'No subscribers' };
      }

      console.log(`📧 Sending blog notification to ${subscribers.length} subscribers...`);

      const emailPromises = subscribers.map(async (subscriber) => {
        const unsubscribeUrl = `${process.env.URL}/api/subscriber/unsubscribe?id=${subscriber.id}&email=${encodeURIComponent(subscriber.email)}`;

        const html = this.#createBlogEmailTemplate(blogData, subscriber, unsubscribeUrl);

        return this.#sendEmail(
          subscriber.email,
          `New Blog Post: ${blogData.title}`,
          html
        );
      });

      const results = await Promise.allSettled(emailPromises);

      const successCount = results.filter(r => r.status === 'fulfilled').length;
      const failCount = results.filter(r => r.status === 'rejected').length;

      console.log(`✅ Blog notification: ${successCount} sent, ${failCount} failed`);

      return {
        success: true,
        sent: successCount,
        failed: failCount,
        total: subscribers.length
      };
    } catch (error) {
      console.error('❌ Error sending blog notifications:', error);
      throw error;
    }
  }

  /**
   * Create enhanced blog email template inspired by the welcome email design
   */
  static #createBlogEmailTemplate(blogData, subscriber, unsubscribeUrl) {
    const {
      title,
      excerpt,
      link,
      imageUrl,
      author,
      publishedDate,
      category = '',
      tags = []
    } = blogData;

    return `
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>${title}</title>
  <style>
    * { margin: 0; padding: 0; box-sizing: border-box; }
    body {
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
      line-height: 1.6;
      color: #1a1a1a;
      background-color: #f5f5f5;
    }
    .email-container {
      max-width: 600px;
      margin: 40px auto;
      background-color: #ffffff;
      border-radius: 8px;
      overflow: hidden;
      box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    }
    .header {
      background-color: #ffffff;
      padding: 32px 40px;
      text-align: center;
      position: relative;
      overflow: hidden;
      border-bottom: 2px solid #f0f0f0;
    }
    .header::before {
      content: '→ → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → → →';
      position: absolute;
      top: 0;
      left: 0;
      width: 200%;
      height: 100%;
      font-size: 20px;
      color: #000000;
      line-height: 40px;
      white-space: nowrap;
      z-index: 0;
      animation: scrollArrows 30s linear infinite;
    }
    @keyframes scrollArrows {
      0% { transform: translateX(0); }
      100% { transform: translateX(-50%); }
    }
    .logo {
      font-size: 26px;
      font-weight: 700;
      color: #000000;
      text-decoration: none;
      letter-spacing: -0.5px;
      position: relative;
      z-index: 1;
    }
    .hero-image {
      width: 100%;
      height: 300px;
      object-fit: cover;
      display: block;
    }
    .content {
      padding: 40px;
    }
    .greeting {
      font-size: 17px;
      color: #000000;
      margin-bottom: 4px;
      font-weight: 500;
    }
    .intro-text {
      font-size: 15px;
      color: #666666;
      margin-bottom: 24px;
      line-height: 1.6;
    }
    .title {
      font-size: 28px;
      font-weight: 700;
      color: #000000;
      margin-bottom: 16px;
      line-height: 1.3;
    }
    .meta {
      display: flex;
      align-items: center;
      gap: 12px;
      margin-bottom: 24px;
      font-size: 14px;
      color: #666666;
    }
    .category {
      display: inline-block;
      padding: 4px 12px;
      background-color: #d0c82a;
      color: #000000;
      border-radius: 4px;
      font-size: 12px;
      font-weight: 600;
      text-transform: uppercase;
      letter-spacing: 0.5px;
    }
    .excerpt {
      font-size: 16px;
      line-height: 1.7;
      color: #333333;
      margin-bottom: 32px;
    }
    .cta-button {
      display: inline-block;
      padding: 14px 32px;
      background-color: #000000;
      color: #ffffff !important;
      text-decoration: none;
      border-radius: 6px;
      font-size: 16px;
      font-weight: 600;
      transition: background-color 0.3s ease;
    }
    .cta-button:hover {
      background-color: #333333;
    }
    .tags {
      margin-top: 32px;
      padding-top: 24px;
      border-top: 1px solid #e5e5e5;
    }
    .tag {
      display: inline-block;
      padding: 6px 12px;
      background-color: #f5f5f5;
      color: #666666;
      border-radius: 4px;
      font-size: 13px;
      margin-right: 8px;
      margin-bottom: 8px;
    }
    .footer {
      background-color: #fafafa;
      padding: 32px 40px;
      text-align: center;
      border-top: 1px solid #e5e5e5;
    }
    .footer-text {
      font-size: 13px;
      color: #999999;
      margin-bottom: 16px;
    }
    .social-links {
      margin-bottom: 16px;
    }
    .social-link {
      display: inline-block;
      margin: 0 8px;
      color: #666666 !important;
      text-decoration: none;
    }
    .unsubscribe {
      font-size: 12px;
      color: #999999;
    }
    .unsubscribe a {
      color: #666666 !important;
      text-decoration: underline;
    }
    @media only screen and (max-width: 600px) {
      .email-container {
        margin: 0;
        border-radius: 0;
      }
      .content {
        padding: 24px;
      }
      .title {
        font-size: 24px;
      }
      .meta {
        flex-direction: column;
        align-items: flex-start;
        gap: 8px;
      }
    }
  </style>
</head>
<body>
  <div class="email-container">
    <!-- Header -->
    <div class="header">
      <a href="https://tlchomes.co.kg/blog" class="logo">Blog Spotlight</a>
    </div>

    <!-- Hero Image -->
    ${imageUrl ? `<img src="${imageUrl}" alt="${title}" class="hero-image">` : ''}

    <!-- Content -->
    <div class="content">
      <p class="greeting">Hi ${subscriber.firstName},</p>
      
      <p class="intro-text">
        We've posted a new blog that you'll be interested in:
      </p>
      
      <h1 class="title">${title}</h1>
      
      <div class="meta">
        ${category ? `<span class="category">${category}</span>` : ''}
        ${publishedDate ? `<span>📅 ${publishedDate}</span>` : ''}
        ${author ? `<span>✍️ ${author}</span>` : ''}
      </div>

      <div class="excerpt">
        ${excerpt}
      </div>

      <a href="${link}" class="cta-button">Read Full Article</a>

      ${tags.length > 0 ? `
        <div class="tags">
          ${tags.map(tag => `<span class="tag">${tag}</span>`).join('')}
        </div>
      ` : ''}
    </div>

    <!-- Footer -->
    <div class="footer">
      <p class="footer-text">
       Stay ahead of the curve with expert perspectives from our top consultants in Kenya's property, agribusiness, procurement, and hospitality industries.
      </p>
      
      <div class="social-links">
        <a href="https://x.com/homes_tlcs" class="social-link">Twitter</a>
        <a href="https://www.facebook.com/p/TLC-HOMES-61565214030939/" class="social-link">Facebook</a>
        <a href="https://www.instagram.com/tlchomes.ke/" class="social-link">Instagram</a>
      </div>

      <p class="footer-text">
        &copy; ${new Date().getFullYear()} TLC Homes Services. All rights reserved.
      </p>
      
      <p class="unsubscribe">
        Don't want to receive these emails? 
        <a href="${unsubscribeUrl}">Unsubscribe here</a>
      </p>
    </div>
  </div>
</body>
</html>
    `;
  }

  /**
   * Verify transporter configuration
   */
  static async verifyConnection() {
    if (!this.transporter) {
      return this.initialize();
    }

    try {
      await this.transporter.verify();
      this.isConnected = true;
      return true;
    } catch (error) {
      this.isConnected = false;
      console.error('❌ Email verification failed:', error.message);
      return false;
    }
  }
}

export default EmailService;