import { randomUUID } from 'crypto';
import EmailService from './emailService.js';
import FileStorage from '../utils/fileStorage.js';
import GoogleSheetsService from './GoogleSheetsService.js'; // ✅ FIXED: Correct import path

/**
 * Subscription service for managing newsletter subscribers
 */
class SubscriptionService {
  /**
   * Subscribe a new user to the newsletter
   * @param {string} email - User email
   * @param {string} firstName - User first name
   * @param {string} lastName - User last name
   * @param {string} source - Subscription source (e.g., 'blog-page')
   */
  static async subscribe(email, firstName, lastName, source = 'website') {
    try {
      // Check if user already exists
      const existingSubscriber = await FileStorage.findByEmail(email);

      if (existingSubscriber) {
        // Check if already verified
        const verifiedSubscribers = await FileStorage.getSubscribers();
        const isVerified = verifiedSubscribers.some(s => s.email === email);

        if (isVerified) {
          return {
            success: false,
            message: 'Email already subscribed and verified',
            status: 'already_subscribed'
          };
        } else {
          return {
            success: false,
            message: 'Verification email already sent. Please check your inbox.',
            status: 'pending_verification'
          };
        }
      }

      // Create new subscriber
      const subscriber = {
        id: randomUUID(),
        email: email.toLowerCase(),
        firstName,
        lastName,
        source,
        subscribedAt: new Date().toISOString(),
        verified: false
      };

      // Add to pending subscribers
      await FileStorage.addPendingSubscriber(subscriber);

      // Send verification email
      await EmailService.sendVerificationEmail(email, subscriber.id, firstName);

      return {
        success: true,
        message: 'Verification email sent. Please check your inbox.',
        status: 'verification_sent',
        subscriberId: subscriber.id
      };
    } catch (error) {
      console.error('Error in subscribe:', error);
      throw error;
    }
  }

  /**
   * Verify a subscriber's email
   * @param {string} id - Subscriber ID
   * @param {string} email - Subscriber email
   */
  static async verifySubscriber(id, email) {
    try {
      // Find pending subscriber
      const pendingSubscriber = await FileStorage.findPendingById(id, email.toLowerCase());

      if (!pendingSubscriber) {
        return {
          success: false,
          message: 'Invalid verification link or already verified',
          status: 'invalid_link'
        };
      }

      // Move from pending to verified subscribers
      const verifiedSubscriber = {
        ...pendingSubscriber,
        verified: true,
        verifiedAt: new Date().toISOString()
      };

      await FileStorage.addSubscriber(verifiedSubscriber);
      await FileStorage.removePendingSubscriber(id);

      console.log(`✅ Subscriber verified: ${email}`);

      // ✅ ADD TO GOOGLE SHEETS
      try {
        console.log('📊 Attempting to add subscriber to Google Sheets...');
        const sheetsResult = await GoogleSheetsService.addSubscriber(verifiedSubscriber);

        if (sheetsResult.success) {
          console.log('✅ Successfully added to Google Sheets');
        } else {
          console.log('⚠️ Failed to add to Google Sheets:', sheetsResult.error);
        }
      } catch (sheetsError) {
        console.error('⚠️ Google Sheets error (non-critical):', sheetsError.message);
        // Don't fail the verification if Google Sheets fails
      }

      return {
        success: true,
        message: 'Email verified successfully! You will now receive our newsletter.',
        status: 'verified',
        subscriber: {
          firstName: verifiedSubscriber.firstName,
          lastName: verifiedSubscriber.lastName,
          email: verifiedSubscriber.email
        }
      };
    } catch (error) {
      console.error('Error in verifySubscriber:', error);
      throw error;
    }
  }

  /**
   * Unsubscribe a user from the newsletter
   * @param {string} id - Subscriber ID
   * @param {string} email - Subscriber email
   */
  static async unsubscribe(id, email) {
    try {
      const subscriber = await FileStorage.findSubscriberById(id, email.toLowerCase());

      if (!subscriber) {
        return {
          success: false,
          message: 'Subscriber not found',
          status: 'not_found'
        };
      }

      // Remove from subscribers
      await FileStorage.removeSubscriber(id);

      console.log(`Subscriber unsubscribed: ${email}`);

      return {
        success: true,
        message: 'You have been successfully unsubscribed from our newsletter.',
        status: 'unsubscribed',
        subscriber: {
          firstName: subscriber.firstName,
          email: subscriber.email
        }
      };
    } catch (error) {
      console.error('Error in unsubscribe:', error);
      throw error;
    }
  }

  /**
   * Get subscriber statistics
   */
  static async getStats() {
    try {
      const subscribers = await FileStorage.getSubscribers();
      const pendingSubscribers = await FileStorage.getPendingSubscribers();

      return {
        verified: subscribers.length,
        pending: pendingSubscribers.length,
        total: subscribers.length + pendingSubscribers.length
      };
    } catch (error) {
      console.error('Error getting stats:', error);
      throw error;
    }
  }

  /**
   * Get all subscribers including verified and unverified (admin only)
   */
  static async getAllSubscribers() {
    try {
      const verifiedSubscribers = await FileStorage.getSubscribers();
      const pendingSubscribers = await FileStorage.getPendingSubscribers();

      // Combine both arrays
      return [...verifiedSubscribers, ...pendingSubscribers];
    } catch (error) {
      console.error('Error getting all subscribers:', error);
      throw error;
    }
  }
}

export default SubscriptionService;