import fs from 'fs/promises';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const STORAGE_PATH = path.join(__dirname, '../../data/subscribers.json');

/**
 * Utility class for managing subscriber data in JSON file
 */
class FileStorage {
  /**
   * Ensure data directory and file exist
   */
  static async ensureStorageExists() {
    try {
      const dataDir = path.dirname(STORAGE_PATH);
      await fs.mkdir(dataDir, { recursive: true });
      
      try {
        await fs.access(STORAGE_PATH);
      } catch {
        // File doesn't exist, create it with empty structure
        await fs.writeFile(STORAGE_PATH, JSON.stringify({
          subscribers: [],
          pendingSubscribers: []
        }, null, 2));
      }
    } catch (error) {
      console.error('Error ensuring storage exists:', error);
      throw error;
    }
  }

  /**
   * Get default data structure
   */
  static getDefaultData() {
    return {
      subscribers: [],
      pendingSubscribers: [],
      lastUpdated: new Date().toISOString()
    };
  }

  /**
   * Read all subscriber data
   */
  static async readData() {
    try {
      await this.ensureStorageExists();
      const data = await fs.readFile(STORAGE_PATH, 'utf8');
      
      // Handle empty file or whitespace-only content
      if (!data || data.trim() === '') {
        console.log('📝 Data file is empty, initializing with default structure');
        const defaultData = this.getDefaultData();
        await this.writeData(defaultData);
        return defaultData;
      }
      
      return JSON.parse(data);
    } catch (error) {
      if (error.code === 'ENOENT') {
        console.log('📝 Data file not found, creating new one');
        const defaultData = this.getDefaultData();
        await this.writeData(defaultData);
        return defaultData;
      }
      
      // Handle JSON parse errors
      if (error instanceof SyntaxError) {
        console.error('❌ Corrupted JSON file, reinitializing:', error.message);
        // Backup corrupted file
        const backupPath = `${STORAGE_PATH}.backup.${Date.now()}`;
        await fs.copyFile(STORAGE_PATH, backupPath).catch(() => {});
        const defaultData = this.getDefaultData();
        await this.writeData(defaultData);
        return defaultData;
      }
      
      throw error;
    }
  }

  /**
   * Write subscriber data to file
   */
  static async writeData(data) {
    try {
      await this.ensureStorageExists();
      await fs.writeFile(STORAGE_PATH, JSON.stringify(data, null, 2));
      return true;
    } catch (error) {
      console.error('Error writing data:', error);
      throw error;
    }
  }

  /**
   * Get all verified subscribers
   */
  static async getSubscribers() {
    const data = await this.readData();
    return data.subscribers || [];
  }

  /**
   * Get all pending subscribers
   */
  static async getPendingSubscribers() {
    const data = await this.readData();
    return data.pendingSubscribers || [];
  }

  /**
   * Add subscriber to verified list
   */
  static async addSubscriber(subscriber) {
    const data = await this.readData();
    data.subscribers.push(subscriber);
    await this.writeData(data);
    return subscriber;
  }

  /**
   * Add subscriber to pending list
   */
  static async addPendingSubscriber(subscriber) {
    const data = await this.readData();
    data.pendingSubscribers.push(subscriber);
    await this.writeData(data);
    return subscriber;
  }

  /**
   * Remove subscriber from pending list
   */
  static async removePendingSubscriber(id) {
    const data = await this.readData();
    data.pendingSubscribers = data.pendingSubscribers.filter(s => s.id !== id);
    await this.writeData(data);
    return true;
  }

  /**
   * Remove subscriber from verified list
   */
  static async removeSubscriber(id) {
    const data = await this.readData();
    data.subscribers = data.subscribers.filter(s => s.id !== id);
    await this.writeData(data);
    return true;
  }

  /**
   * Find subscriber by email in either list
   */
  static async findByEmail(email) {
    const data = await this.readData();
    const verified = data.subscribers.find(s => s.email === email);
    const pending = data.pendingSubscribers.find(s => s.email === email);
    return verified || pending || null;
  }

  /**
   * Find pending subscriber by ID and email
   */
  static async findPendingById(id, email) {
    const data = await this.readData();
    return data.pendingSubscribers.find(s => s.id === id && s.email === email) || null;
  }

  /**
   * Find verified subscriber by ID and email
   */
  static async findSubscriberById(id, email) {
    const data = await this.readData();
    return data.subscribers.find(s => s.id === id && s.email === email) || null;
  }
}

export default FileStorage;
